<?php

namespace App\Http\Controllers\Admin\Inventory;

use App\Enums\GlobalEnum;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rules;
use DataTables;
use App\Models\Inventory\InventoryCategory;
use App\Models\Inventory\InventoryItemBrand;
use App\Models\Inventory\InventoryItemTax;
use App\Models\Inventory\InventoryUnit;
use App\Models\Inventory\InventoryAttribute;
use App\Models\Inventory\InventoryAttributeValue;
use App\Models\Inventory\InventoryItem;
use App\Models\Inventory\InventoryItemSkuAttributeValue;
use App\Models\Inventory\InventoryItemSku;
use App\Services\Inventory\InventoryService;
use App\Services\Inventory\SaleableItemService;
use App\Http\Requests\Inventory\SalabelItemFormRequest;

class InventoryController extends Controller
{

    private InventoryService $inventoryService;
    private SaleableItemService $saleableItemService;
    public function __construct(InventoryService $inventoryService, SaleableItemService $saleableItemService)
    {
        $this->inventoryService = $inventoryService;
        $this->saleableItemService = $saleableItemService;
    }

    public function addNewSaleableItem()
    {
        $categories = InventoryCategory::active()->where('parent_id', 1)->get();
        $brands = InventoryItemBrand::active()->get();
        $taxes = InventoryItemTax::active()->get();
        $units = InventoryUnit::active()->get();
        $attributes = InventoryAttribute::active()->get();
        $attributeValue = InventoryAttribute::with('values')->get();
        return view('admin.inventory.addNew-saleableItem', compact('categories', 'brands', 'taxes', 'units', 'attributes', 'attributeValue'));
    }

    public function findAttributeValueByAttribute(Request $request)
    {
        $output = '';

        $values = InventoryAttributeValue::active()
            ->whereIn('inventory_attribute_id', $request->selected)
            ->get();

        $attributes = $values->groupBy('inventory_attribute_id');

        // Load selected values if editing
        $selectedValues = [];

        if ($request->filled('inventory_item_id')) {
            // Get all SKU IDs for this product
            $skuIds = InventoryItemSku::where('inventory_item_id', $request->inventory_item_id)->pluck('id');

            // Get all attribute values tied to those SKUs
            $selectedValues = InventoryItemSkuAttributeValue::whereIn('inventory_item_sku_id', $skuIds)
                ->pluck('inventory_attribute_value_id')
                ->toArray();
        }


        foreach ($attributes as $attributeId => $attributeValues) {
            $optionName = InventoryAttribute::find($attributeId)->name;

            $output .= '
            <div class="xl:col-span-3">
                <input type="text" class="form-control" value="'.$optionName.'" readonly>
            </div>
            <div class="xl:col-span-9">
                <select class="form-control combination-input" id="'.$optionName.'" data-choices="" data-choices-removeitem="" name="'.$optionName.'-values[]" data-type="'.$optionName.'" multiple="">
        ';

            foreach ($attributeValues as $value) {
                $isSelected = in_array($value->id, $selectedValues) ? 'selected' : '';
                $output .= '<option value="'.$value->id.'" '.$isSelected.'>'.$value->value.'</option>';
            }

            $output .= '
                </select>
            </div>
        ';
        }
        echo $output;

    }

    public function generateCombinations(Request $request)
    {
        $attributes = $request->input('attributes', []);
        $existingSkus = $request->input('existingSkus', []);

        // If less than two attributes, flatten and use only labels
        if (count($attributes) < 2) {
            $combinations = collect($attributes)->flatten(1)->pluck('label')->all();
            return $this->inventoryService->generateCombinationTable($combinations, $existingSkus);
        }

        // Generate full combinations
        $combinations = $this->inventoryService->generateAttributeCombinations($attributes);
        return $this->inventoryService->generateCombinationTable($combinations, $existingSkus);
    }

    public function getCategories($id)
    {
        $data = InventoryCategory::active()->where('parent_id', $id)->select('id','name')->get();
        return response()->json($data);
    }


   /**
    * saleable inventory
    */

    public function storeSaleableItem(SalabelItemFormRequest $request)
    {
        try {
            $this->saleableItemService->store($request);
            return response()->json(['message' => 'Success'], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => $e->getMessage()], 500);
        }

    }

    public function editItem($code)
    {
        $product = InventoryItem::with('category','attributes','skus')->where('code', $code)->firstOrFail();
        $brands = InventoryItemBrand::active()->get();
        $taxes = InventoryItemTax::active()->get();
        $units = InventoryUnit::active()->get();
        $attributes = InventoryAttribute::active()->get();
        $attributeValue = InventoryAttribute::with('values')->get();
        //category
        $selectedCategory = $product->category;
        $categoryPath = [];
        $current = $selectedCategory;

        while ($current instanceof InventoryCategory) {
            $categoryPath[] = $current;
            // Stop if current level is "Head"
            if ($current->category_level === 'Group') {
                break;
            }
            $current = $current->parent;
        }
        $categoryPath = array_reverse($categoryPath);
        $productAttributes = $product->attributes;

        $existingSkus = $product->skus->mapWithKeys(function ($sku) {
            return [
                $sku->sku_code => [
                    'price' => $sku->selling_price,
                    'id' => $sku->id,
                ]
            ];
        });

        return view('admin.inventory.addNew-saleableItem', compact(
    'categoryPath',
            'brands',
            'taxes',
            'units',
            'attributes',
            'attributeValue',
            'product',
            'productAttributes',
            'existingSkus',
        ));
    }

}
