<?php

namespace App\Models\Inventory;

use App\Enums\InventoryEnum;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class InventoryItem extends Model
{
    use HasFactory;
    protected $fillable = [
        'code',
        'slug',
        'title',
        'category_id',
        'brand_id',
        'is_variant',
        'sales_price',
        'purchase_unit',
        'sales_unit',
        'unit_conversion',
        'item_type',
        'purchase_type',
        'serial_no',
        'lot_no',
        'hscode',
        'chosen_attribute',
        'choice_options',
        'chosen_attribute_options',
        'chosen_attribute_option_values',
        'variations',
        'description',
        'discount',
        'discount_type',
        'discount_start_date',
        'discount_end_date',
        'taxes_id',
        'item_img',
        'published',
        'serial',
    ];

    protected static function boot()
    {
        parent::boot();
        static::creating(function ($item) {
            if (empty($unit->code)) {
                $maxId = InventoryItem::max('id');
                $nextNumber = $maxId ? intval($maxId) + 1 : 1;
                $item->code = str_pad($nextNumber, 6, '0', STR_PAD_LEFT);
            }
            if (empty($item->url)) {
                $slug = Str::slug($item->title);
                $item->slug = self::makeUniqueSlug($slug);
            }

        });
    }

    public static function makeUniqueSlug($slug, $excludeId = null)
    {
        $originalSlug = $slug;
        $counter = 1;

        while (self::slugExists($slug, $excludeId)) {
            $slug = "{$originalSlug}-{$counter}";
            $counter++;
        }

        return $slug;
    }

    private static function slugExists($slug, $excludeId = null)
    {
        $query = InventoryItem::where('slug', $slug);

        if ($excludeId) {
            $query->where('id', '!=', $excludeId);
        }

        return $query->exists();
    }

    // MANY-TO-MANY: Product <-> Attribute
    public function attributes()
    {
        return $this->belongsToMany(InventoryAttribute::class, 'inventory_item_attributes', 'inventory_item_id', 'inventory_attribute_id');
    }
    public function skus()
    {
        return $this->hasMany(InventoryItemSku::class);
    }

    public function category()
    {
        return $this->belongsTo(InventoryCategory::class, 'category_id');
    }




}
