<?php

namespace App\Models\Inventory;

use App\Enums\InventoryEnum;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class InventoryItemBrand extends Model
{
    use HasFactory;
    protected $fillable = [
        'code',
        'name',
        'logo',
        'slug',
        'meta_title',
        'meta_description',
        'is_active',
        'serial',
    ];

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($brand) {
            if (empty($brand->code)) {
                $maxGroup = InventoryItemBrand::max('code');
                $nextNumber = $maxGroup ? intval($maxGroup) + 1 : 1;
                $brand->code = str_pad($nextNumber, 4, '0', STR_PAD_LEFT);
            };

            // Generate URL from name if empty
            if (empty($brand->slug)) {
                $slug = Str::slug($brand->name);
                $brand->slug = self::makeUniqueSlug($slug);
            }
        });
    }

    private static function makeUniqueSlug($slug, $excludeId = null)
    {
        $originalSlug = $slug;
        $counter = 1;

        while (self::slugExists($slug, $excludeId)) {
            $slug = "{$originalSlug}-{$counter}";
            $counter++;
        }

        return $slug;
    }

    private static function slugExists($slug, $excludeId = null)
    {
        $query = self::where('slug', $slug);

        if ($excludeId) {
            $query->where('id', '!=', $excludeId);
        }

        return $query->exists();
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', 'Active');
    }

}
