<?php

namespace App\Services\Inventory;
use App\Models\Inventory\InventoryCategory;
use App\Models\Inventory\InventoryAttribute;
use App\Models\Inventory\InventoryAttributeValue;
use App\Models\Inventory\InventoryItem;
use App\Models\Inventory\InventoryItemSkuAttributeValue;
use App\Models\Inventory\InventoryItemSku;
use App\Http\Requests\Inventory\SalabelItemFormRequest;
use Illuminate\Support\Facades\DB;
use App\Helpers\FileUploadHelper;

class SaleableItemService
{

    public function store(SalabelItemFormRequest $request)
    {
        DB::beginTransaction();

        try {
            $data = $request->validated();
            $data['category_id'] = $request->input('final_category_id');
            $data['is_variant'] = $request->has('is_variant') ? 1 : 0;
            $data['item_type'] = 'Salabel';

            //Handle the file upload
            if ($request->hasFile('item_img')) {
                $filename = FileUploadHelper::uploadImage($request->file('item_img'));
                $data['item_img'] = $filename;
            }

            $product = InventoryItem::updateOrCreate(
                ['id' => $request->id],
                $data
            );

            if($data['is_variant']){
                //Variant Item

                // Attach attributes
                $product->attributes()->sync($request->options);

                // Build attribute-value groups
                $attributeValueGroups = [];
                foreach ($request->options as $attributeId) {
                    $attribute = InventoryAttribute::findOrFail($attributeId);
                    $fieldName = $attribute->name . '-values';
                    $values = $request->input($fieldName);

                    if (!empty($values)) {
                        $attributeValueGroups[] = $values;
                    }
                }

                // Generate combinations
                $combinations = collect(array_shift($attributeValueGroups));
                foreach ($attributeValueGroups as $group) {
                    $combinations = $combinations->crossJoin($group)->map(function ($item) {
                        return is_array($item) ? $item : [$item];
                    });
                }

                //get all skus Ids
                $skuIds = $product->skus()->pluck('id');
                if ($skuIds->isNotEmpty()) {
                    InventoryItemSkuAttributeValue::whereIn('inventory_item_sku_id', $skuIds)->delete();
                }

                foreach ($combinations as $index => $combination) {
                    $skuId = $request->skuIds[$index] ?? null;

                    $skuData = [
                        'sku_code' => $request->variants[$index],
                        'selling_price' => $request->prices[$index],
                        'is_variant' => $request->has('is_variant') ? 1 : 0
                    ];

                    // Build the attribute_value_map for this SKU
                    $attributeValueMap = [];

                    foreach ($combination as $i => $attributeValueId) {
                        $attributeId = $request->options[$i];
                        $attributeValueMap[] = [
                            'attribute_id' => $attributeId,
                            'value_id' => $attributeValueId,
                        ];
                    }

                    // Add the map as JSON
                    $skuData['attribute_value_map'] = $attributeValueMap;

                    // Create or update the SKU
                    if ($skuId) {
                        $sku = $product->skus()->where('id', $skuId)->first();
                        if ($sku) {
                            $sku->update($skuData);
                        } else {
                            $sku = $product->skus()->create($skuData);
                        }
                    } else {
                        $sku = $product->skus()->create($skuData);
                    }

                    foreach ($combination as $i => $attributeValueId) {
                        $attributeId = $request->options[$i];

                        InventoryItemSkuAttributeValue::create([
                            'inventory_item_sku_id' => $sku->id,
                            'inventory_attribute_id' => $attributeId,
                            'inventory_attribute_value_id' => $attributeValueId,
                        ]);

                    }
                    //\Log::info("SKU $sku->id valid attribute value IDs:", $existingIds);
                }


            }else {
                //Single Item
                $sku = InventoryItemSku::where('inventory_item_id', $product->id)->where('is_variant', 0)->first();

                $skuData = [
                    'sku_code' => null,
                    'selling_price' => $request->input('sales_price'),
                    'is_variant' => $request->has('is_variant') ? 1 : 0
                ];

                // Create or update the SKU
                    if ($sku) {
                        $sku->update($skuData);
                    } else {
                        $sku = $product->skus()->create($skuData);
                    }

            };



            DB::commit();
            return true;

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }




}
