<?php

namespace Database\Seeders;

use App\Models\Admin;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class RolePermissionSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Clear cache
        app()[\Spatie\Permission\PermissionRegistrar::class]->forgetCachedPermissions();

        // Define permissions
        $permissions = [
            // Group: Dashboard
            [
                'group_name' => 'Dashboard',
                'permissions' => [
                    'access dashboard',
                ]
            ],

            // Group: User Management
            [
                'group_name' => 'User Management',
                'permissions' => [
                    'create user',
                    'view user',
                    'edit user',
                    'delete user',
                ]
            ],

            // Group: Role Management
            [
                'group_name' => 'Role Management',
                'permissions' => [
                    'create role',
                    'view role',
                    'edit role',
                    'delete role',
                ]
            ],

            // Group: Product Management
            [
                'group_name' => 'Product Management',
                'permissions' => [
                    'create product',
                    'view product',
                    'edit product',
                    'delete product',
                ]
            ],

            // Group: Post Management
            [
                'group_name' => 'Post Management',
                'permissions' => [
                    'create post',
                    'view post',
                    'edit post',
                    'delete post',
                ]
            ],

            // Group: Settings
            [
                'group_name' => 'Settings',
                'permissions' => [
                    'access settings',
                ]
            ],
        ];

        // Loop through and create permissions
        $guard = 'admin'; // or 'web'

        // Create permissions
        foreach ($permissions as $group) {
            foreach ($group['permissions'] as $permissionName) {
                Permission::firstOrCreate([
                    'name' => $permissionName,
                    'guard_name' => $guard,
                    'group_name' => $group['group_name'],
                ]);
            }
        }

        // Define roles and assign permissions
        $superAdmin = Role::firstOrCreate(['name' => 'Super Admin', 'guard_name' => $guard]);
        $admin      = Role::firstOrCreate(['name' => 'Admin', 'guard_name' => $guard]);
        $manager    = Role::firstOrCreate(['name' => 'Manager', 'guard_name' => $guard]);
        $editor     = Role::firstOrCreate(['name' => 'Editor', 'guard_name' => $guard]);
        $user       = Role::firstOrCreate(['name' => 'User', 'guard_name' => $guard]);

        // Super Admin gets all permissions
        $superAdmin->syncPermissions(Permission::all());

        // Admin role gets most permissions except settings
        $admin->syncPermissions([
            'access dashboard',

            'create user',
            'view user',
            'edit user',

            'create role',
            'view role',
            'edit role',

            'create product',
            'view product',
            'edit product',
            'delete product',

            'create post',
            'view post',
            'edit post',
            'delete post',
        ]);

        // Manager role can manage products and view users
        $manager->syncPermissions([
            'access dashboard',
            'view user',

            'create product',
            'view product',
            'edit product',
        ]);

        // Editor can only manage posts
        $editor->syncPermissions([
            'access dashboard',

            'create post',
            'view post',
            'edit post',
        ]);

        // User can only view posts/products
        $user->syncPermissions([
            'view post',
            'view product',
        ]);

        // Optional: Assign roles to specific users
        $this->assignRolesToUsers();

        $this->command->info('Roles and permissions successfully seeded!');
    }

    private function assignRolesToUsers()
    {
        $admin = Admin::find(1);

        // Just for demo, first user = Super Admin, etc.
        if ($admin->count()) {
            $admin->assignRole('Super Admin');

            $this->command->info('Demo roles assigned to users.');
        } else {
            $this->command->warn('No users found to assign roles. Seed users first!');
        }
    }
}
